import sys
import re

class AbyssColors:
    # Standard 16 colors
    BLACK = "\033[30m"
    RED = "\033[31m"
    GREEN = "\033[32m"
    YELLOW = "\033[33m"
    BLUE = "\033[34m"
    MAGENTA = "\033[35m"
    CYAN = "\033[36m"
    WHITE = "\033[37m"
    
    # Bright 16 colors
    BRIGHT_BLACK = "\033[90m"
    BRIGHT_RED = "\033[91m"
    BRIGHT_GREEN = "\033[92m"
    BRIGHT_YELLOW = "\033[93m"
    BRIGHT_BLUE = "\033[94m"
    BRIGHT_MAGENTA = "\033[95m"
    BRIGHT_CYAN = "\033[96m"
    BRIGHT_WHITE = "\033[97m"

    # Style
    BOLD = "\033[1m"
    FAINT = "\033[2m"
    ITALIC = "\033[3m"
    UNDERLINE = "\033[4m"
    BLINK = "\033[5m"
    INVERSE = "\033[7m"
    STRIKE = "\033[9m"
    RESET = "\033[0m"

    # 256 Color Helpers
    @staticmethod
    def fg(n): return f"\033[38;5;{n}m"
    
    @staticmethod
    def bg(n): return f"\033[48;5;{n}m"

    # Cyber-Crimson 256 Palette (Primary Theme)
    DEEP_RED = fg(52)
    DARK_RED = fg(88)
    BLOOD_RED = fg(124)
    RED = fg(160)
    NEON_RED = fg(196)
    VOID_WHITE = fg(255)
    DEEP_GREY = fg(237)
    SOFT_GREY = fg(244)

    # Re-mapped common names for red-themed standards
    PRIMARY = NEON_RED
    SECONDARY = BLOOD_RED
    ACCENT = DARK_RED
    ERROR = PLASMA_RED = fg(196)
    
    # Keeping others as secondary options
    NEON_PINK = fg(198)
    NEON_BLUE = fg(45)
    NEON_CYAN = fg(51)
    NEON_GREEN = fg(121)

    # Regex to catch ANSI escape sequences
    _ANSI_ESCAPE = re.compile(r'\033\[[0-9;]*m')

    @classmethod
    def strip(cls, text):
        """Remove ANSI escape sequences to get 'visible' length."""
        return cls._ANSI_ESCAPE.sub('', text)

class AbyssText:
    # Preset Borders
    STYLES = {
        "box": "┌─┐││└─┘",
        "double": "╔═╗║║╚═╝",
        "rounded": "╭─╮││╰─╯",
        "dotted": "┌┄┐┆┆└┄┘",
        "heavy": "┏━┓┃┃┗━┛",
    }
    
    _CURRENT_THEME = {
        "primary": AbyssColors.PRIMARY,
        "secondary": AbyssColors.SECONDARY,
        "outline": AbyssColors.SECONDARY
    }

    @classmethod
    def set_theme(cls, theme_name):
        """
        Sets the global theme. 
        Options: "red", "green", "blue", "cyan", "purple", "white"
        """
        themes = {
            "red": {
                "primary": AbyssColors.NEON_RED,
                "secondary": AbyssColors.BLOOD_RED,
                "outline": AbyssColors.DARK_RED
            },
            "green": {
                "primary": fg(121), # Neon Green
                "secondary": fg(46), # Matrix Green
                "outline": fg(22)   # Deep Green
            },
            "blue": {
                "primary": fg(45),  # Neon Blue
                "secondary": fg(33), # Dodgy Blue
                "outline": fg(18)   # Dark Blue
            },
            "cyan": {
                "primary": fg(51),  # Neon Cyan
                "secondary": fg(44), # Aqua
                "outline": fg(30)   # Deep Cyan
            },
            "purple": {
                "primary": fg(171), # Neon Purple/Pink
                "secondary": fg(135), # Cyber Purple
                "outline": fg(129)   # Deep Purple
            },
            "white": {
                "primary": AbyssColors.VOID_WHITE,
                "secondary": AbyssColors.SOFT_GREY,
                "outline": AbyssColors.DEEP_GREY
            }
        }
        
        if theme_name in themes:
            cls._CURRENT_THEME = themes[theme_name]

    @staticmethod
    def _apply_style(text, color):
        if color:
            return f"{color}{text}{AbyssColors.RESET}"
        return text

    @staticmethod
    def _get_border_chars(outline_char):
        if not outline_char:
            return None
        if outline_char in AbyssText.STYLES:
            outline_char = AbyssText.STYLES[outline_char]
        if len(outline_char) == 8:
            return {
                'tl': outline_char[0], 't': outline_char[1], 'tr': outline_char[2],
                'l': outline_char[3], 'r': outline_char[4],
                'bl': outline_char[5], 'b': outline_char[6], 'br': outline_char[7]
            }
        return {
            'tl': outline_char[0], 't': outline_char, 'tr': outline_char[-1],
            'l': outline_char, 'r': outline_char,
            'bl': outline_char[0], 'b': outline_char, 'br': outline_char[-1],
            'is_pattern': True
        }

    @staticmethod
    def _get_pattern_char(pattern, index):
        if not pattern:
            return " "
        return pattern[index % len(pattern)]

    @classmethod
    def print(cls, *args, 
              sep=" ", 
              end="\n", 
              color=None, 
              outline_char=None, 
              outline_color=None, 
              padding=1, 
              width=None):
        """Enhanced print with global theme support."""
        content = sep.join(map(str, args))
        lines = content.split('\n')
        
        # Apply theme defaults
        if color is None: color = cls._CURRENT_THEME["primary"]
        if outline_color is None: outline_color = cls._CURRENT_THEME["outline"]

        if outline_char is True:
            outline_char = "box"

        if not outline_char:
            for i, line in enumerate(lines):
                c = color[i] if isinstance(color, (list, tuple)) and i < len(color) else color
                sys.stdout.write(cls._apply_style(line, c) + ("\n" if i < len(lines)-1 else ""))
            sys.stdout.write(end)
            return

        b = cls._get_border_chars(outline_char)
        is_pattern = b.get('is_pattern', False)
        line_colors = color if isinstance(color, (list, tuple)) else [color] * len(lines)
        if len(line_colors) < len(lines):
            line_colors.extend([None] * (len(lines) - len(line_colors)))

        max_line_len = max(len(AbyssColors.strip(line)) for line in lines)
        if width: max_line_len = max(max_line_len, width - 2 - (padding * 2))
        box_width = max_line_len + 2 + (padding * 2)
        output = []
        
        # Top
        top = "".join(cls._get_pattern_char(outline_char, i) for i in range(box_width)) if is_pattern else b['tl'] + (b['t'] * (box_width - 2)) + b['tr']
        output.append(cls._apply_style(top, outline_color))
        
        # Pad Top
        for _ in range(padding):
            side_l, side_r = (b['l'][0], b['r'][-1]) if is_pattern else (b['l'], b['r'])
            output.append(cls._apply_style(side_l, outline_color) + (" " * (box_width - 2)) + cls._apply_style(side_r, outline_color))
        
        # Lines
        for i, line in enumerate(lines):
            side_l, side_r = (b['l'][0], b['r'][-1]) if is_pattern else (b['l'], b['r'])
            visible_text = AbyssColors.strip(line)
            needed_padding = max_line_len - len(visible_text)
            styled_text = cls._apply_style(line, line_colors[i])
            inner = (" " * padding) + styled_text + (" " * needed_padding) + (" " * padding)
            output.append(cls._apply_style(side_l, outline_color) + inner + cls._apply_style(side_r, outline_color))
            
        # Pad Bottom
        for _ in range(padding):
            side_l, side_r = (b['l'][0], b['r'][-1]) if is_pattern else (b['l'], b['r'])
            output.append(cls._apply_style(side_l, outline_color) + (" " * (box_width - 2)) + cls._apply_style(side_r, outline_color))
            
        # Bottom
        bottom = "".join(cls._get_pattern_char(outline_char, i) for i in range(box_width)) if is_pattern else b['bl'] + (b['b'] * (box_width - 2)) + b['br']
        output.append(cls._apply_style(bottom, outline_color))
        sys.stdout.write("\n".join(output) + end)

    @classmethod
    def input(cls, prompt="", 
              color=None, 
              input_color=None, 
              outline_char=None, 
              outline_color=None, 
              padding=1,
              split=False):
        """Styled input with global theme support."""
        if color is None: color = cls._CURRENT_THEME["primary"]
        if input_color is None: input_color = AbyssColors.VOID_WHITE

        if outline_char:
            # Boxed prompt
            cls.print(prompt, color=color, outline_char=outline_char, 
                      outline_color=outline_color, padding=padding, end="")
            sys.stdout.write("\n> " if split else " ")
        else:
            # Standard inline or split prompt
            sys.stdout.write(cls._apply_style(prompt, color))
            if split:
                sys.stdout.write("\n> ")
        
        if input_color: sys.stdout.write(input_color)
        user_input = input()
        sys.stdout.write(AbyssColors.RESET)
        return user_input

# Alias
abyss_print = AbyssText.print
abyss_input = AbyssText.input
def set_theme(n): AbyssText.set_theme(n)
def fg(n): return AbyssColors.fg(n)

if __name__ == "__main__":
    set_theme("green")
    abyss_print("MATRIX THEME ACTIVE", outline_char="double")
    val = abyss_input("Target identified. Proceed?", split=True)
    abyss_print(f"Executing protocol for {val}...")

if __name__ == "__main__":
    # Examples
    print("--- Basic Colors ---")
    abyss_print("Hello from the Abyss!", color=AbyssColors.BRIGHT_CYAN)
    
    print("\n--- Outlined Text ---")
    abyss_print("System Initialized", 
                color=AbyssColors.GREEN, 
                outline_char="#", 
                outline_color=AbyssColors.BRIGHT_BLACK)
    
    print("\n--- Patterned Outline ---")
    abyss_print("WARNING: BREACH DETECTED", 
                color=AbyssColors.BRIGHT_RED, 
                outline_char="!*", 
                outline_color=AbyssColors.YELLOW,
                padding=2)
    
    print("\n--- Multi-line with Custom Outline ---")
    abyss_print("Status: Active\nUptime: 100%\nNodes: 12", 
                color=AbyssColors.BRIGHT_WHITE, 
                outline_char="=+", 
                outline_color=AbyssColors.BLUE)

    print("\n--- Default ASCII style (using specific char) ---")
    abyss_print("ABYSS TEXT ENGINE", 
                color=AbyssColors.MAGENTA, 
                outline_char="+", 
                outline_color=AbyssColors.MAGENTA)
