import socket
import concurrent.futures
import ipaddress
from abyss_text import abyss_print, abyss_input, AbyssColors, set_theme

# Common ports that are often points of interest or vulnerability
VULNERABLE_PORTS = {
    21: "FTP (File Transfer Protocol) - Often prone to brute force.",
    22: "SSH (Secure Shell) - Potential for unauthorized access.",
    23: "Telnet - Unencrypted, highly reaching vulnerability.",
    25: "SMTP (Mail Transfer) - Vulnerable to relay attacks.",
    53: "DNS (Domain Name System) - Potential for cache poisoning.",
    80: "HTTP (Web) - Standard web traffic, many attack vectors.",
    110: "POP3 (Email) - Often unencrypted.",
    135: "RPC (Remote Procedure Call) - System level access point.",
    139: "NetBIOS - Windows networking vulnerability point.",
    443: "HTTPS (Secure Web) - SSL/TLS attack surface.",
    445: "SMB (Server Message Block) - High risk for ransomware/worms.",
    1433: "SQL Server - Database injection/brute force target.",
    3306: "MySQL - Database access point.",
    3389: "RDP (Remote Desktop) - High target for remote takeover.",
    5432: "Postgres - Database access point.",
    8080: "HTTP Proxy/Alt - Common alternate web port.",
}

def get_local_ip():
    """Returns the primary local IP address of the machine."""
    s = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)
    try:
        s.connect(('8.8.8.8', 1))
        ip = s.getsockname()[0]
    except Exception:
        ip = '127.0.0.1'
    finally:
        s.close()
    return ip

def discover_hosts(subnet, max_workers=255):
    """Checks for active hosts on a given subnet by attempting a quick connection."""
    active_hosts = []
    
    def check_host(ip):
        # Try a few common ports to see if host is alive - Ultra Fast
        for port in [80, 443, 22, 135, 445]:
            try:
                with socket.socket(socket.AF_INET, socket.SOCK_STREAM) as s:
                    s.settimeout(0.1) # Aggressive
                    if s.connect_ex((str(ip), port)) == 0:
                        return str(ip)
            except:
                pass
        return None

    network = ipaddress.ip_network(subnet, strict=False)
    with concurrent.futures.ThreadPoolExecutor(max_workers=max_workers) as executor:
        futures = [executor.submit(check_host, ip) for ip in network.hosts()]
        for future in concurrent.futures.as_completed(futures):
            res = future.result()
            if res:
                active_hosts.append(res)
                
    return active_hosts

def grab_banner(s):
    """Attempts to read a service banner from an open socket."""
    try:
        s.settimeout(0.5) # Reduced
        banner = s.recv(1024).decode(errors='ignore').strip()
        return banner
    except:
        return None

def check_specific_vulnerabilities(ip, port, banner):
    """Performs deep checks for specific vulnerabilities."""
    findings = []
    
    if port == 21:
        try:
            with socket.socket(socket.AF_INET, socket.SOCK_STREAM) as s:
                s.settimeout(0.8) # Aggressive
                s.connect((ip, port))
                s.recv(1024)
                s.send(b"USER anonymous\r\n")
                s.recv(1024)
                s.send(b"PASS a@b.com\r\n")
                res = s.recv(1024).decode(errors='ignore')
                if "230" in res: findings.append("CRITICAL: Anonymous FTP")
        except: pass

    if port in [80, 8080]:
        try:
            with socket.socket(socket.AF_INET, socket.SOCK_STREAM) as s:
                s.settimeout(0.5)
                s.connect((ip, port))
                s.send(b"HEAD / HTTP/1.0\r\n\r\n")
                res = s.recv(1024).decode(errors='ignore')
                if "Server:" in res:
                    server_header = res.split('Server:')[1].split('\n')[0].strip()
                    findings.append(f"INFO: {server_header}")
        except: pass

    if port == 23: findings.append("HIGH: Telnet")
    return findings

def scan_single_port(ip, port, timeout=0.08):
    """Checks if a single port is open and grabs info/vulns - Red Team Speed."""
    try:
        with socket.socket(socket.AF_INET, socket.SOCK_STREAM) as s:
            s.settimeout(timeout)
            result = s.connect_ex((ip, port))
            if result == 0:
                banner = grab_banner(s)
                vulns = check_specific_vulnerabilities(ip, port, banner)
                return {"port": port, "banner": banner, "vulns": vulns}
    except: pass
    return None

def deep_port_scan(target="127.0.0.1", port_range=None, max_workers=1000):
    """
    Scans ports at extreme speed.
    """
    if port_range is None: ports_to_scan = range(1, 1025)
    elif isinstance(port_range, tuple): ports_to_scan = range(port_range[0], port_range[1] + 1)
    else: ports_to_scan = port_range

    results_data = []

    # High worker count for TOTAL ANNIHILATION
    with concurrent.futures.ThreadPoolExecutor(max_workers=max_workers) as executor:
        future_to_port = {executor.submit(scan_single_port, target, port): port for port in ports_to_scan}
        for future in concurrent.futures.as_completed(future_to_port):
            data = future.result()
            if data:
                port = data['port']
                desc = VULNERABLE_PORTS.get(port, "Open Port")
                results_data.append({
                    "port": port,
                    "description": desc,
                    "vulnerable": port in VULNERABLE_PORTS or len(data['vulns']) > 0,
                    "banner": data['banner'],
                    "vulns": data['vulns']
                })

    return sorted(results_data, key=lambda x: x['port'])

def run_standalone():
    set_theme("red")
    abyss_print("◢ ABYSS_STRIKE v4.0: OMNISCIENCE ◣", outline_char="double")
    
    local_ip = get_local_ip()
    abyss_print(f"LOCAL INTERFACE: {local_ip}", color=AbyssColors.ACCENT)
    
    target_mode = abyss_input("TARGET (1: Local, 2: Remote, 3: Subnet, 4: TOTAL ANNIHILATION): ")
    
    targets = []
    ports_to_scan = None

    if target_mode == "1":
        targets = ["127.0.0.1"]
    elif target_mode == "2":
        targets = [abyss_input("ENTER REMOTE IP: ")]
    elif target_mode == "3" or target_mode == "4":
        # Robustly calculate the /24 subnet (e.g., 10.0.0.13 -> 10.0.0.0/24)
        base_ip = local_ip.rpartition('.')[0]
        default_subnet = f"{base_ip}.0/24"
        
        subnet = abyss_input(f"SUBNET (Default {default_subnet}): ") or default_subnet
        abyss_print(f"COMMENCING HOST DISCOVERY ON {subnet}...", color=AbyssColors.ACCENT)
        
        try:
            targets = discover_hosts(subnet)
            if target_mode == "4":
                if "127.0.0.1" not in targets: targets.append("127.0.0.1")
            abyss_print(f"DISCOVERED {len(targets)} ACTIVE HOSTS.", color=AbyssColors.NEON_GREEN)
        except Exception as e:
            abyss_print(f"ERROR: Invalid Network Definition ({e})", color=AbyssColors.PLASMA_RED)
            return
    
    if not targets:
        abyss_print("NO VALID TARGETS IDENTIFIED.", color=AbyssColors.PLASMA_RED)
        return

    if target_mode == "4":
        # Total Annihilation mode: Everything, every port.
        mode = "2" # Force Full Scan
    else:
        mode = abyss_input("SCAN TYPE (1: Fast, 2: Full [1-65535], 3: Vulnerable Only): ")
    
    if mode == "1": scan_range = range(1, 1025)
    elif mode == "2": scan_range = range(1, 65535)
    else: scan_range = list(VULNERABLE_PORTS.keys())

    abyss_print(f"! COMMENCING TOTAL AUDIT OF {len(targets)} HOSTS !", 
                color=AbyssColors.BOLD + AbyssColors.PLASMA_RED,
                outline_char="heavy",
                padding=1)

    for target in targets:
        abyss_print(f"\n--- AUDITING {target} ---", color=AbyssColors.NEON_CYAN)
        results = deep_port_scan(target, scan_range)
        
        if not results:
            abyss_print(f"[{target}] NO VULNERABILITIES FOUND.", color=AbyssColors.DEEP_GREY)
            continue

        for r in results:
            is_serious = any("CRITICAL" in v or "HIGH" in v for v in r['vulns']) or r['vulnerable']
            color = AbyssColors.PLASMA_RED if is_serious else AbyssColors.VOID_WHITE
            
            output = [f"PORT {r['port']} | {r['description']}"]
            if r['banner']: output.append(f"BANNER: {r['banner']}")
            for v in r['vulns']: output.append(f"FINDING: {v}")
            
            abyss_print("\n".join(output), 
                        color=color, 
                        outline_char="heavy" if is_serious else "box",
                        padding=1)

if __name__ == "__main__":
    run_standalone()
